/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common.io;

import java.io.EOFException;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.eclipse.jetty.io.AbstractConnection;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.Scheduler;
import org.eclipse.jetty.websocket.api.BatchMode;
import org.eclipse.jetty.websocket.api.CloseException;
import org.eclipse.jetty.websocket.api.CloseStatus;
import org.eclipse.jetty.websocket.api.SuspendToken;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.WriteCallback;
import org.eclipse.jetty.websocket.api.extensions.ExtensionConfig;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.common.CloseInfo;
import org.eclipse.jetty.websocket.common.ConnectionState;
import org.eclipse.jetty.websocket.common.Generator;
import org.eclipse.jetty.websocket.common.LogicalConnection;
import org.eclipse.jetty.websocket.common.Parser;
import org.eclipse.jetty.websocket.common.WebSocketSession;
import org.eclipse.jetty.websocket.common.io.FrameFlusher;
import org.eclipse.jetty.websocket.common.io.IOState;

public abstract class AbstractWebSocketConnection
extends AbstractConnection
implements LogicalConnection,
IOState.ConnectionStateListener,
Dumpable {
    private static final Logger LOG = Log.getLogger(AbstractWebSocketConnection.class);
    private static final int MIN_BUFFER_SIZE = 28;
    private final ByteBufferPool bufferPool;
    private final Scheduler scheduler;
    private final Generator generator;
    private final Parser parser;
    private final WebSocketPolicy policy;
    private final AtomicBoolean suspendToken;
    private final FrameFlusher flusher;
    private WebSocketSession session;
    private List<ExtensionConfig> extensions;
    private boolean isFilling;
    private IOState ioState;
    private Stats stats = new Stats();

    public AbstractWebSocketConnection(EndPoint endp, Executor executor, Scheduler scheduler, WebSocketPolicy policy, ByteBufferPool bufferPool) {
        super(endp, executor, true);
        this.policy = policy;
        this.bufferPool = bufferPool;
        this.generator = new Generator(policy, bufferPool);
        this.parser = new Parser(policy, bufferPool);
        this.scheduler = scheduler;
        this.extensions = new ArrayList<ExtensionConfig>();
        this.suspendToken = new AtomicBoolean(false);
        this.ioState = new IOState();
        this.ioState.addListener(this);
        this.flusher = new Flusher(bufferPool, this.generator, endp);
        this.setInputBufferSize(policy.getInputBufferSize());
        this.setMaxIdleTimeout(policy.getIdleTimeout());
    }

    @Override
    public Executor getExecutor() {
        return super.getExecutor();
    }

    @Override
    public void close() {
        this.close(1000, null);
    }

    @Override
    public void close(int statusCode, String reason) {
        CloseInfo close = new CloseInfo(statusCode, reason);
        if (statusCode == 1006) {
            this.flusher.close();
            this.ioState.onAbnormalClose(close);
        } else {
            this.ioState.onCloseLocal(close);
        }
    }

    @Override
    public void disconnect() {
        LOG.debug("{} disconnect()", new Object[]{this.policy.getBehavior()});
        this.flusher.close();
        this.disconnect(false);
    }

    private void disconnect(boolean onlyOutput) {
        EndPoint endPoint = this.getEndPoint();
        LOG.debug("Shutting down output {}", endPoint);
        endPoint.shutdownOutput();
        if (!onlyOutput) {
            LOG.debug("Closing {}", endPoint);
            endPoint.close();
        }
    }

    protected void execute(Runnable task) {
        try {
            this.getExecutor().execute(task);
        }
        catch (RejectedExecutionException e) {
            LOG.debug("Job not dispatched: {}", task);
        }
    }

    @Override
    public void fillInterested() {
        this.stats.countFillInterestedEvents.incrementAndGet();
        super.fillInterested();
    }

    @Override
    public ByteBufferPool getBufferPool() {
        return this.bufferPool;
    }

    public List<ExtensionConfig> getExtensions() {
        return this.extensions;
    }

    public Generator getGenerator() {
        return this.generator;
    }

    @Override
    public long getIdleTimeout() {
        return this.getEndPoint().getIdleTimeout();
    }

    @Override
    public IOState getIOState() {
        return this.ioState;
    }

    @Override
    public long getMaxIdleTimeout() {
        return this.getEndPoint().getIdleTimeout();
    }

    public Parser getParser() {
        return this.parser;
    }

    @Override
    public WebSocketPolicy getPolicy() {
        return this.policy;
    }

    @Override
    public InetSocketAddress getRemoteAddress() {
        return this.getEndPoint().getRemoteAddress();
    }

    public Scheduler getScheduler() {
        return this.scheduler;
    }

    @Override
    public WebSocketSession getSession() {
        return this.session;
    }

    public Stats getStats() {
        return this.stats;
    }

    @Override
    public boolean isOpen() {
        return this.getIOState().isOpen() && this.getEndPoint().isOpen();
    }

    @Override
    public boolean isReading() {
        return this.isFilling;
    }

    @Override
    public void onClose() {
        super.onClose();
        this.flusher.close();
    }

    @Override
    public void onConnectionStateChange(ConnectionState state) {
        LOG.debug("{} Connection State Change: {}", new Object[]{this.policy.getBehavior(), state});
        switch (state) {
            case OPEN: {
                LOG.debug("fillInterested", new Object[0]);
                this.fillInterested();
                break;
            }
            case CLOSED: {
                if (this.ioState.wasAbnormalClose()) {
                    CloseInfo abnormal = new CloseInfo(1001, "Abnormal Close - " + this.ioState.getCloseInfo().getReason());
                    this.outgoingFrame(abnormal.asFrame(), new OnDisconnectCallback(), BatchMode.OFF);
                    break;
                }
                this.disconnect();
                break;
            }
            case CLOSING: {
                CloseInfo close = this.ioState.getCloseInfo();
                this.outgoingFrame(close.asFrame(), new OnDisconnectCallback(), BatchMode.OFF);
            }
        }
    }

    @Override
    public void onFillable() {
        LOG.debug("{} onFillable()", new Object[]{this.policy.getBehavior()});
        this.stats.countOnFillableEvents.incrementAndGet();
        ByteBuffer buffer = this.bufferPool.acquire(this.getInputBufferSize(), true);
        boolean readMore = false;
        try {
            this.isFilling = true;
            readMore = this.read(buffer) != -1;
        }
        finally {
            this.bufferPool.release(buffer);
        }
        if (readMore && !this.suspendToken.get()) {
            this.fillInterested();
        } else {
            this.isFilling = false;
        }
    }

    @Override
    protected void onFillInterestedFailed(Throwable cause) {
        LOG.ignore(cause);
        this.stats.countFillInterestedEvents.incrementAndGet();
        super.onFillInterestedFailed(cause);
    }

    @Override
    public void onOpen() {
        super.onOpen();
        this.ioState.onOpened();
    }

    @Override
    protected boolean onReadTimeout() {
        LOG.debug("{} Read Timeout", new Object[]{this.policy.getBehavior()});
        IOState state = this.getIOState();
        if (state.getConnectionState() == ConnectionState.CLOSING || state.getConnectionState() == ConnectionState.CLOSED) {
            return true;
        }
        this.session.notifyError(new SocketTimeoutException("Timeout on Read"));
        this.close(1006, "Idle Timeout");
        return false;
    }

    @Override
    public void outgoingFrame(Frame frame, WriteCallback callback, BatchMode batchMode) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("outgoingFrame({}, {})", frame, callback);
        }
        this.flusher.enqueue(frame, callback, batchMode);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private int read(ByteBuffer buffer) {
        EndPoint endPoint = this.getEndPoint();
        try {
            while (true) {
                int filled;
                if ((filled = endPoint.fill(buffer)) == 0) {
                    return 0;
                }
                if (filled < 0) {
                    LOG.debug("read - EOF Reached (remote: {})", this.getRemoteAddress());
                    this.ioState.onReadEOF();
                    return -1;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Filled {} bytes - {}", filled, BufferUtil.toDetailString(buffer));
                }
                this.parser.parse(buffer);
            }
        }
        catch (IOException e) {
            LOG.warn(e);
            this.close(1002, e.getMessage());
            return -1;
        }
        catch (CloseException e) {
            LOG.warn(e);
            this.close(e.getStatusCode(), e.getMessage());
            return -1;
        }
    }

    @Override
    public void resume() {
        if (this.suspendToken.getAndSet(false)) {
            this.fillInterested();
        }
    }

    public void setExtensions(List<ExtensionConfig> extensions) {
        this.extensions = extensions;
    }

    @Override
    public void setInputBufferSize(int inputBufferSize) {
        if (inputBufferSize < 28) {
            throw new IllegalArgumentException("Cannot have buffer size less than 28");
        }
        super.setInputBufferSize(inputBufferSize);
    }

    @Override
    public void setMaxIdleTimeout(long ms) {
        this.getEndPoint().setIdleTimeout(ms);
    }

    @Override
    public void setSession(WebSocketSession session) {
        this.session = session;
    }

    @Override
    public SuspendToken suspend() {
        this.suspendToken.set(true);
        return this;
    }

    @Override
    public String dump() {
        return ContainerLifeCycle.dump(this);
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        out.append(this.toString()).append(System.lineSeparator());
    }

    @Override
    public String toString() {
        return String.format("%s{f=%s,g=%s,p=%s}", super.toString(), this.flusher, this.generator, this.parser);
    }

    private class Flusher
    extends FrameFlusher {
        private Flusher(ByteBufferPool bufferPool, Generator generator, EndPoint endpoint) {
            super(bufferPool, generator, endpoint, AbstractWebSocketConnection.this.getPolicy().getMaxBinaryMessageBufferSize(), 8);
        }

        @Override
        protected void onFailure(Throwable x) {
            if (AbstractWebSocketConnection.this.ioState.wasAbnormalClose()) {
                LOG.ignore(x);
                return;
            }
            LOG.debug("Write flush failure", x);
            String reason = "Websocket write failure";
            if (x instanceof EOFException) {
                reason = "EOF";
                Throwable cause = x.getCause();
                if (cause != null && StringUtil.isNotBlank(cause.getMessage())) {
                    reason = "EOF: " + cause.getMessage();
                }
            } else if (StringUtil.isNotBlank(x.getMessage())) {
                reason = x.getMessage();
            }
            reason = CloseStatus.trimMaxReasonLength(reason);
            AbstractWebSocketConnection.this.session.notifyError(x);
            AbstractWebSocketConnection.this.session.notifyClose(1006, reason);
            AbstractWebSocketConnection.this.disconnect();
        }
    }

    public class OnDisconnectCallback
    implements WriteCallback {
        @Override
        public void writeFailed(Throwable x) {
            AbstractWebSocketConnection.this.disconnect();
        }

        @Override
        public void writeSuccess() {
            AbstractWebSocketConnection.this.disconnect();
        }
    }

    public static class Stats {
        private AtomicLong countFillInterestedEvents = new AtomicLong(0L);
        private AtomicLong countOnFillableEvents = new AtomicLong(0L);
        private AtomicLong countFillableErrors = new AtomicLong(0L);

        public long getFillableErrorCount() {
            return this.countFillableErrors.get();
        }

        public long getFillInterestedCount() {
            return this.countFillInterestedEvents.get();
        }

        public long getOnFillableCount() {
            return this.countOnFillableEvents.get();
        }
    }
}

