/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.MQTTConnection;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Session {
    private static final Logger LOG = LoggerFactory.getLogger(Session.class);
    private static final int FLIGHT_BEFORE_RESEND_MS = 5000;
    private static final int INFLIGHT_WINDOW_SIZE = 10;
    private final String clientId;
    private boolean clean;
    private Will will;
    private Queue<SessionRegistry.EnqueuedMessage> sessionQueue;
    private final AtomicReference<SessionStatus> status = new AtomicReference<SessionStatus>(SessionStatus.DISCONNECTED);
    private MQTTConnection mqttConnection;
    private List<Subscription> subscriptions = new ArrayList<Subscription>();
    private final Map<Integer, SessionRegistry.EnqueuedMessage> inflightWindow = new HashMap<Integer, SessionRegistry.EnqueuedMessage>();
    private final DelayQueue<InFlightPacket> inflightTimeouts = new DelayQueue();
    private final Map<Integer, MqttPublishMessage> qos2Receiving = new HashMap<Integer, MqttPublishMessage>();
    private final AtomicInteger inflightSlots = new AtomicInteger(10);

    Session(String clientId, boolean clean, Will will, Queue<SessionRegistry.EnqueuedMessage> sessionQueue) {
        this(clean, clientId, sessionQueue);
        this.will = will;
    }

    Session(boolean clean, String clientId, Queue<SessionRegistry.EnqueuedMessage> sessionQueue) {
        this.clientId = clientId;
        this.clean = clean;
        this.sessionQueue = sessionQueue;
    }

    void update(boolean clean, Will will) {
        this.clean = clean;
        this.will = will;
    }

    void markConnected() {
        this.assignState(SessionStatus.DISCONNECTED, SessionStatus.CONNECTED);
    }

    void bind(MQTTConnection mqttConnection) {
        this.mqttConnection = mqttConnection;
    }

    public boolean disconnected() {
        return this.status.get() == SessionStatus.DISCONNECTED;
    }

    public boolean connected() {
        return this.status.get() == SessionStatus.CONNECTED;
    }

    public String getClientID() {
        return this.clientId;
    }

    public List<Subscription> getSubscriptions() {
        return new ArrayList<Subscription>(this.subscriptions);
    }

    public void addSubscriptions(List<Subscription> newSubscriptions) {
        this.subscriptions.addAll(newSubscriptions);
    }

    public boolean hasWill() {
        return this.will != null;
    }

    public Will getWill() {
        return this.will;
    }

    boolean assignState(SessionStatus expected, SessionStatus newState) {
        return this.status.compareAndSet(expected, newState);
    }

    public void closeImmediately() {
        this.mqttConnection.dropConnection();
    }

    public void disconnect() {
        boolean res = this.assignState(SessionStatus.CONNECTED, SessionStatus.DISCONNECTING);
        if (!res) {
            return;
        }
        this.mqttConnection = null;
        this.will = null;
        this.assignState(SessionStatus.DISCONNECTING, SessionStatus.DISCONNECTED);
    }

    boolean isClean() {
        return this.clean;
    }

    public void processPubRec(int packetId) {
        this.inflightWindow.remove(packetId);
        this.inflightSlots.incrementAndGet();
        if (this.canSkipQueue()) {
            this.inflightSlots.decrementAndGet();
            int pubRelPacketId = packetId;
            this.inflightWindow.put(pubRelPacketId, new SessionRegistry.PubRelMarker());
            this.inflightTimeouts.add(new InFlightPacket(pubRelPacketId, 5000L));
            MqttMessage pubRel = MQTTConnection.pubrel(pubRelPacketId);
            this.mqttConnection.sendIfWritableElseDrop(pubRel);
            this.drainQueueToConnection();
        } else {
            this.sessionQueue.add(new SessionRegistry.PubRelMarker());
        }
    }

    public void processPubComp(int messageID) {
        this.inflightWindow.remove(messageID);
        this.inflightSlots.incrementAndGet();
        this.drainQueueToConnection();
    }

    public void sendPublishOnSessionAtQos(Topic topic, MqttQoS qos, ByteBuf payload) {
        switch (qos) {
            case AT_MOST_ONCE: {
                if (!this.connected()) break;
                this.mqttConnection.sendPublishNotRetainedQos0(topic, qos, payload);
                break;
            }
            case AT_LEAST_ONCE: {
                this.sendPublishQos1(topic, qos, payload);
                break;
            }
            case EXACTLY_ONCE: {
                this.sendPublishQos2(topic, qos, payload);
                break;
            }
            case FAILURE: {
                LOG.error("Not admissible");
            }
        }
    }

    private void sendPublishQos1(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (!this.connected() && this.isClean()) {
            return;
        }
        if (this.canSkipQueue()) {
            this.inflightSlots.decrementAndGet();
            int packetId = this.mqttConnection.nextPacketId();
            this.inflightWindow.put(packetId, new SessionRegistry.PublishedMessage(topic, qos, payload));
            this.inflightTimeouts.add(new InFlightPacket(packetId, 5000L));
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(topic.toString(), qos, payload, packetId);
            this.mqttConnection.sendPublish(publishMsg);
        } else {
            SessionRegistry.PublishedMessage msg = new SessionRegistry.PublishedMessage(topic, qos, payload);
            this.sessionQueue.add(msg);
        }
    }

    private void sendPublishQos2(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (this.canSkipQueue()) {
            this.inflightSlots.decrementAndGet();
            int packetId = this.mqttConnection.nextPacketId();
            this.inflightWindow.put(packetId, new SessionRegistry.PublishedMessage(topic, qos, payload));
            this.inflightTimeouts.add(new InFlightPacket(packetId, 5000L));
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(topic.toString(), qos, payload, packetId);
            this.mqttConnection.sendPublish(publishMsg);
            this.drainQueueToConnection();
        } else {
            SessionRegistry.PublishedMessage msg = new SessionRegistry.PublishedMessage(topic, qos, payload);
            this.sessionQueue.add(msg);
        }
    }

    private boolean canSkipQueue() {
        return this.sessionQueue.isEmpty() && this.inflightSlots.get() > 0 && this.connected() && this.mqttConnection.channel.isWritable();
    }

    void pubAckReceived(int ackPacketId) {
        this.inflightWindow.remove(ackPacketId);
        this.inflightSlots.incrementAndGet();
        this.drainQueueToConnection();
    }

    public void resendInflightNotAcked() {
        ArrayList<InFlightPacket> expired = new ArrayList<InFlightPacket>(10);
        this.inflightTimeouts.drainTo(expired);
        this.debugLogPacketIds(expired);
        for (InFlightPacket notAckPacketId : expired) {
            if (!this.inflightWindow.containsKey(notAckPacketId.packetId)) continue;
            SessionRegistry.PublishedMessage msg = (SessionRegistry.PublishedMessage)this.inflightWindow.get(notAckPacketId.packetId);
            Topic topic = msg.topic;
            MqttQoS qos = msg.publishingQos;
            ByteBuf payload = msg.payload;
            ByteBuf copiedPayload = payload.retainedDuplicate();
            MqttPublishMessage publishMsg = this.publishNotRetainedDuplicated(notAckPacketId, topic, qos, copiedPayload);
            this.mqttConnection.sendPublish(publishMsg);
        }
    }

    private void debugLogPacketIds(Collection<InFlightPacket> expired) {
        if (!LOG.isDebugEnabled() || expired.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        for (InFlightPacket packet : expired) {
            sb.append(packet.packetId).append(", ");
        }
        LOG.debug("Resending {} in flight packets [{}]", (Object)expired.size(), (Object)sb);
    }

    private MqttPublishMessage publishNotRetainedDuplicated(InFlightPacket notAckPacketId, Topic topic, MqttQoS qos, ByteBuf payload) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, true, qos, false, 0);
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic.toString(), notAckPacketId.packetId);
        return new MqttPublishMessage(fixedHeader, varHeader, payload);
    }

    private void drainQueueToConnection() {
        while (!this.canSkipQueue()) {
            SessionRegistry.EnqueuedMessage msg = this.sessionQueue.remove();
            this.inflightSlots.decrementAndGet();
            int sendPacketId = this.mqttConnection.nextPacketId();
            this.inflightWindow.put(sendPacketId, msg);
            if (msg instanceof SessionRegistry.PubRelMarker) {
                MqttMessage pubRel = MQTTConnection.pubrel(sendPacketId);
                this.mqttConnection.sendIfWritableElseDrop(pubRel);
                continue;
            }
            SessionRegistry.PublishedMessage msgPub = (SessionRegistry.PublishedMessage)msg;
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(msgPub.topic.toString(), msgPub.publishingQos, msgPub.payload, sendPacketId);
            this.mqttConnection.sendPublish(publishMsg);
        }
    }

    public void writabilityChanged() {
        this.drainQueueToConnection();
    }

    public void sendQueuedMessagesWhileOffline() {
        LOG.trace("Republishing all saved messages for session {} on CId={}", (Object)this, (Object)this.clientId);
        this.drainQueueToConnection();
    }

    void sendRetainedPublishOnSessionAtQos(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (qos != MqttQoS.AT_MOST_ONCE) {
            this.mqttConnection.sendPublishRetainedWithPacketId(topic, qos, payload);
        } else {
            this.mqttConnection.sendPublishRetainedQos0(topic, qos, payload);
        }
    }

    public void receivedPublishQos2(int messageID, MqttPublishMessage msg) {
        this.qos2Receiving.put(messageID, msg);
        msg.retain();
        this.mqttConnection.sendPublishReceived(messageID);
    }

    public void receivedPubRelQos2(int messageID) {
        MqttPublishMessage removedMsg = this.qos2Receiving.remove(messageID);
        ReferenceCountUtil.release((Object)removedMsg);
    }

    public String toString() {
        return "Session{clientId='" + this.clientId + '\'' + ", clean=" + this.clean + ", status=" + this.status + ", inflightSlots=" + this.inflightSlots + '}';
    }

    static class InFlightPacket
    implements Delayed {
        final int packetId;
        private long startTime;

        InFlightPacket(int packetId, long delayInMilliseconds) {
            this.packetId = packetId;
            this.startTime = System.currentTimeMillis() + delayInMilliseconds;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long diff = this.startTime - System.currentTimeMillis();
            return unit.convert(diff, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            if (this.startTime - ((InFlightPacket)o).startTime == 0L) {
                return 0;
            }
            if (this.startTime - ((InFlightPacket)o).startTime > 0L) {
                return 1;
            }
            return -1;
        }
    }

    static enum SessionStatus {
        CONNECTED,
        CONNECTING,
        DISCONNECTING,
        DISCONNECTED;

    }

    static final class Will {
        final String topic;
        final ByteBuf payload;
        final MqttQoS qos;
        final boolean retained;

        Will(String topic, ByteBuf payload, MqttQoS qos, boolean retained) {
            this.topic = topic;
            this.payload = payload;
            this.qos = qos;
            this.retained = retained;
        }
    }
}

